using System;
using System.Collections.Generic;
using System.Globalization;
using System.Web.Routing;
using Nop.Core;
using Nop.Core.Domain.Directory;
using Nop.Core.Domain.Orders;
using Nop.Core.Domain.Payments;
using Nop.Core.Domain.Shipping;
using Nop.Core.Plugins;
using Nop.Plugin.Payments.BasisPay.Controllers;
using Nop.Services.Configuration;
using Nop.Services.Directory;
using Nop.Services.Localization;
using Nop.Services.Payments;
using Nop.Web.Framework;
using System.Web;
using Nop.Services.Logging;

namespace Nop.Plugin.Payments.BasisPay
{
    /// <summary>
    /// BasisPay payment processor
    /// </summary>
    public class BasisPayPaymentProcessor : BasePlugin, IPaymentMethod
    {
        #region Fields

        private readonly BasisPayPaymentSettings _BasisPayPaymentSettings;
        private readonly ISettingService _settingService;
        private readonly ICurrencyService _currencyService;
        private readonly CurrencySettings _currencySettings;
        private readonly IWebHelper _webHelper;
        private readonly IStoreContext _storeContext;
        private readonly HttpContextBase _httpContext;
        private readonly ILogger _logger;
        private readonly ILocalizationService _localizationService;

        #endregion

        #region Ctor

        public BasisPayPaymentProcessor(BasisPayPaymentSettings BasisPayPaymentSettings,
            ISettingService settingService, ICurrencyService currencyService, ILogger logger, ILocalizationService localizationService,
            CurrencySettings currencySettings, IWebHelper webHelper, IStoreContext storeContext, HttpContextBase httpContext)
        {
            this._BasisPayPaymentSettings = BasisPayPaymentSettings;
            this._settingService = settingService;
            this._currencyService = currencyService;
            this._currencySettings = currencySettings;
            this._webHelper = webHelper;
            this._storeContext = storeContext;
            this._httpContext = httpContext;
            this._logger = logger;
            this._localizationService = localizationService;
        }

        #endregion

        #region Utilities

        #endregion

        #region Methods

        /// <summary>
        /// Process a payment
        /// </summary>
        /// <param name="processPaymentRequest">Payment info required for an order processing</param>
        /// <returns>Process payment result</returns>
        public ProcessPaymentResult ProcessPayment(ProcessPaymentRequest processPaymentRequest)
        {
            var result = new ProcessPaymentResult();
            result.NewPaymentStatus = PaymentStatus.Pending;
            return result;
        }

        /// <summary>
        /// Post process payment (used by payment gateways that require redirecting to a third-party URL)
        /// </summary>
        /// <param name="postProcessPaymentRequest">Payment info required for an order processing</param>
        public void PostProcessPayment(PostProcessPaymentRequest postProcessPaymentRequest)
        {
            var myUtility = new BasisPayHelper();

            var orderId = postProcessPaymentRequest.Order.Id;
            var remotePostHelper = new RemotePost();
            decimal amount = postProcessPaymentRequest.Order.OrderTotal;
            amount = Math.Round(amount, 2);

            string[] hash_columns = {
            "address_line_1",
            "address_line_2",
            "amount",
            "api_key",
            "city",
            "country",
            "currency",
            "description",
            "email",
            "mode",
            "name",
            "order_id",
            "phone",
            "return_url",
            "state",
            "udf1",
            "udf2",
            "udf3",
            "udf4",
            "udf5",
            "zip_code"
            };
            IDictionary<string, string> data = new Dictionary<string, string>();

            data["salt"] = _BasisPayPaymentSettings.Salt.ToString();
            data["api_key"] = _BasisPayPaymentSettings.Api_Key.ToString();
            data["mode"] = _BasisPayPaymentSettings.TransactMode.ToString();
            data["return_url"] = _webHelper.GetStoreLocation(false) + "Plugins/PaymentBasisPay/Return";
            data["order_id"] = orderId.ToString();
            data["currency"] = _currencyService.GetCurrencyById(_currencySettings.PrimaryStoreCurrencyId).CurrencyCode;
            data["description"] = "BASISPAYV2_NopCommerceV3.7: Paying invoice " + orderId.ToString() + " of store " + _storeContext.CurrentStore.Name + " using nop commerce plugin";
            data["name"] = postProcessPaymentRequest.Order.BillingAddress.FirstName.ToString();
            data["address_line_1"] = postProcessPaymentRequest.Order.BillingAddress.Address1;
            data["address_line_2"] = postProcessPaymentRequest.Order.BillingAddress.Address2;
            data["city"] = postProcessPaymentRequest.Order.BillingAddress.City;
            data["state"] = postProcessPaymentRequest.Order.BillingAddress.StateProvince.Name;
            data["zip_code"] = postProcessPaymentRequest.Order.BillingAddress.ZipPostalCode;
            data["country"] = postProcessPaymentRequest.Order.BillingAddress.Country.ThreeLetterIsoCode;
            data["amount"] = amount.ToString();
            data["phone"] = postProcessPaymentRequest.Order.BillingAddress.PhoneNumber;
            data["email"] = postProcessPaymentRequest.Order.BillingAddress.Email.ToString();
            data["udf1"] = _httpContext.Request.ServerVariables["REMOTE_ADDR"];  //IP address 


            remotePostHelper.FormName = "BasisPayForm";
            remotePostHelper.Url = "https://pay.basispay.in/v2/paymentrequest";

            remotePostHelper.Add("api_key", data["api_key"]);
            remotePostHelper.Add("mode", data["mode"]);
            remotePostHelper.Add("return_url", data["return_url"]);
            remotePostHelper.Add("order_id", data["order_id"]);
            remotePostHelper.Add("currency", data["currency"]);
            remotePostHelper.Add("description", data["description"]);
            remotePostHelper.Add("name", data["name"]);
            remotePostHelper.Add("address_line_1", data["address_line_1"]);
            remotePostHelper.Add("address_line_2", data["address_line_2"]);
            remotePostHelper.Add("city", data["city"]);
            remotePostHelper.Add("state", data["state"]);
            remotePostHelper.Add("zip_code", data["zip_code"]);
            remotePostHelper.Add("country", data["country"]);
            remotePostHelper.Add("amount", data["amount"]);
            remotePostHelper.Add("phone", data["phone"]);
            remotePostHelper.Add("email", data["email"]);
            remotePostHelper.Add("udf1", data["udf1"]);

            string[] hash_values = myUtility.gethash(hash_columns, data);

            hash_values[0] = hash_values[0].Remove(0, 40);
            hash_values[0] = hash_values[0].Insert(0, "****************SALT********************");
            _logger.Information("Hash data sent to the server  : " + hash_values[0]);
            remotePostHelper.Add("hash", hash_values[1]);
            remotePostHelper.Post();
        }



        //Hide payment begins

        public bool HidePaymentMethod(IList<ShoppingCartItem> cart)
        {
            //you can put any logic here
            //for example, hide this payment method if all products in the cart are downloadable
            //or hide this payment method if current customer is from certain country
            return false;
        }

        //hide payment ends

        /// <summary>
        /// Gets additional handling fee
        /// </summary>
        /// <param name="cart">Shoping cart</param>
        /// <returns>Additional handling fee</returns>
        public decimal GetAdditionalHandlingFee(IList<ShoppingCartItem> cart)
        {
            return 0;
        }

        /// <summary>
        /// Captures payment
        /// </summary>
        /// <param name="capturePaymentRequest">Capture payment request</param>
        /// <returns>Capture payment result</returns>
        public CapturePaymentResult Capture(CapturePaymentRequest capturePaymentRequest)
        {
            var result = new CapturePaymentResult();
            result.AddError("Capture method not supported");
            return result;
        }

        /// <summary>
        /// Refunds a payment
        /// </summary>
        /// <param name="refundPaymentRequest">Request</param>
        /// <returns>Result</returns>
        public RefundPaymentResult Refund(RefundPaymentRequest refundPaymentRequest)
        {
            var result = new RefundPaymentResult();
            result.AddError("Refund method not supported");
            return result;
        }

        /// <summary>
        /// Voids a payment
        /// </summary>
        /// <param name="voidPaymentRequest">Request</param>
        /// <returns>Result</returns>
        public VoidPaymentResult Void(VoidPaymentRequest voidPaymentRequest)
        {
            var result = new VoidPaymentResult();
            result.AddError("Void method not supported");
            return result;
        }

        /// <summary>
        /// Process recurring payment
        /// </summary>
        /// <param name="processPaymentRequest">Payment info required for an order processing</param>
        /// <returns>Process payment result</returns>
        public ProcessPaymentResult ProcessRecurringPayment(ProcessPaymentRequest processPaymentRequest)
        {
            var result = new ProcessPaymentResult();
            result.AddError("Recurring payment not supported");
            return result;
        }

        /// <summary>
        /// Cancels a recurring payment
        /// </summary>
        /// <param name="cancelPaymentRequest">Request</param>
        /// <returns>Result</returns>
        public CancelRecurringPaymentResult CancelRecurringPayment(CancelRecurringPaymentRequest cancelPaymentRequest)
        {
            var result = new CancelRecurringPaymentResult();
            result.AddError("Recurring payment not supported");
            return result;
        }

        /// <summary>
        /// Gets a value indicating whether customers can complete a payment after order is placed but not completed (for redirection payment methods)
        /// </summary>
        /// <param name="order">Order</param>
        /// <returns>Result</returns>
        public bool CanRePostProcessPayment(Order order)
        {
            if (order == null)
                throw new ArgumentNullException("order");

            //BasisPay is the redirection payment method
            //It also validates whether order is also paid (after redirection) so customers will not be able to pay twice

            //payment status should be Pending
            if (order.PaymentStatus != PaymentStatus.Pending)
                return false;

            //let's ensure that at least 1 minute passed after order is placed
            if ((DateTime.UtcNow - order.CreatedOnUtc).TotalMinutes < 1)
                return false;

            return true;
        }

        /// <summary>
        /// Gets a route for provider configuration
        /// </summary>
        /// <param name="actionName">Action name</param>
        /// <param name="controllerName">Controller name</param>
        /// <param name="routeValues">Route values</param>
        public void GetConfigurationRoute(out string actionName, out string controllerName, out RouteValueDictionary routeValues)
        {
            actionName = "Configure";
            controllerName = "PaymentBasisPay";
            routeValues = new RouteValueDictionary() { { "Namespaces", "Nop.Plugin.Payments.BasisPay.Controllers" }, { "area", null } };
        }

        /// <summary>
        /// Gets a route for payment info
        /// </summary>
        /// <param name="actionName">Action name</param>
        /// <param name="controllerName">Controller name</param>
        /// <param name="routeValues">Route values</param>
        public void GetPaymentInfoRoute(out string actionName, out string controllerName, out RouteValueDictionary routeValues)
        {
            actionName = "PaymentInfo";
            controllerName = "PaymentBasisPay";
            routeValues = new RouteValueDictionary() { { "Namespaces", "Nop.Plugin.Payments.BasisPay.Controllers" }, { "area", null } };
        }

        public Type GetControllerType()
        {
            return typeof(PaymentBasisPayController);
        }

        public override void Install()
        {
            var settings = new BasisPayPaymentSettings()
            {
                Api_Key = "",
                Salt = "",
            };
            _settingService.SaveSetting(settings);

            //locales
            this.AddOrUpdatePluginLocaleResource("Plugins.Payments.BasisPay.RedirectionTip", "You will be redirected to BasisPay site to complete the order.");
            this.AddOrUpdatePluginLocaleResource("Plugins.Payments.BasisPay.Api_Key", "Key");
            this.AddOrUpdatePluginLocaleResource("Plugins.Payments.BasisPay.Api_Key.Hint", "Enter Key.");
            this.AddOrUpdatePluginLocaleResource("Plugins.Payments.BasisPay.Salt", "Salt");
            this.AddOrUpdatePluginLocaleResource("Plugins.Payments.BasisPay.Salt.Hint", "Enter salt.");
            this.AddOrUpdatePluginLocaleResource("Plugins.Payments.BasisPay.TransactMode", "Transaction mode");
            this.AddOrUpdatePluginLocaleResource("Plugins.Payments.BasisPay.TransactMode.Hint", "Choose transaction mode");
            this.AddOrUpdatePluginLocaleResource("Plugins.Payments.BasisPay.PaymentMethodDescription", "Pay by Basispay");


            base.Install();
        }

        public override void Uninstall()
        {
            //locales
            this.DeletePluginLocaleResource("Plugins.Payments.BasisPay.RedirectionTip");
            this.DeletePluginLocaleResource("Plugins.Payments.BasisPay.Api_Key");
            this.DeletePluginLocaleResource("Plugins.Payments.BasisPay.Api_Key.Hint");
            this.DeletePluginLocaleResource("Plugins.Payments.BasisPay.Salt");
            this.DeletePluginLocaleResource("Plugins.Payments.BasisPay.Salt.Hint");
            this.DeletePluginLocaleResource("Plugins.Payments.BasisPay.TransactMode");
            this.DeletePluginLocaleResource("Plugins.Payments.BasisPay.TransactMode.Hint");
            this.DeletePluginLocaleResource("Plugins.Payments.BasisPay.PaymentMethodDescription");


            base.Uninstall();
        }
        #endregion

        #region Properties

        /// <summary>
        /// Gets a value indicating whether capture is supported
        /// </summary>
        public bool SupportCapture
        {
            get
            {
                return false;
            }
        }

        /// <summary>
        /// Gets a value indicating whether partial refund is supported
        /// </summary>
        public bool SupportPartiallyRefund
        {
            get
            {
                return false;
            }
        }

        /// <summary>
        /// Gets a value indicating whether refund is supported
        /// </summary>
        public bool SupportRefund
        {
            get
            {
                return false;
            }
        }

        /// <summary>
        /// Gets a value indicating whether void is supported
        /// </summary>
        public bool SupportVoid
        {
            get
            {
                return false;
            }
        }

        /// <summary>
        /// Gets a recurring payment type of payment method
        /// </summary>
        public RecurringPaymentType RecurringPaymentType
        {
            get
            {
                return RecurringPaymentType.NotSupported;
            }
        }

        /// <summary>
        /// Gets a payment method type
        /// </summary>
        public PaymentMethodType PaymentMethodType
        {
            get
            {
                return PaymentMethodType.Redirection;
            }
        }


        public bool SkipPaymentInfo
        {
            get
            {
                return false;
            }
        }

        /// <summary>
        /// Gets a payment method description that will be displayed on checkout pages in the public store
        /// </summary>
        public string PaymentMethodDescription
        {
            //return description of this payment method to be display on "payment method" checkout step. good practice is to make it localizable
            //for example, for a redirection payment method, description may be like this: "You will be redirected to PayPal site to complete the payment"
            get { return _localizationService.GetResource("Plugins.Payments.BasisPay.PaymentMethodDescription"); }
        }

        #endregion
    }
}