<?php
// vim: set ts=4 sw=4 sts=4 et:

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */


namespace XLite\Module\XC\Basispay\Model\Payment\Processor;

/**
 * Basispay payment processor
 */
class Basispay extends \XLite\Model\Payment\Base\WebBased
{
    /**
     * Production form URL
     */
    const PRODUCTION_FORM_URL = 'https://pay.basispay.in/v2/paymentrequest';

    /**
     * Test form URL
     */
    const TEST_FORM_URL = 'https://pay.basispay.in/v2/paymentrequest';

    /**
     * Get settings widget or template
     *
     * @return string Widget class name or template path
     */
    public function getSettingsWidget()
    {
        return 'modules/XC/Basispay/config.twig';
    }

    /**
     * Process return
     *
     * @param \XLite\Model\Payment\Transaction $transaction Return-owner transaction
     *
     * @return void
     */
    public function processReturn(\XLite\Model\Payment\Transaction $transaction)
    {
        static::log(
            array('request' => \XLite\Core\Request::getInstance()->getData())
        );

        parent::processReturn($transaction);

        $request = \XLite\Core\Request::getInstance();
        $status = $transaction::STATUS_SUCCESS;
	    $hash_columns=$request->getPostData();
	    unset($hash_columns['hash']);
	    ksort($hash_columns);

	    $private_salt = $this->getSetting('salt');
	    $hash_data = $private_salt;

	    foreach ($hash_columns as $key=>$value) {
		    if (strlen($value) > 0) {
			    $hash_data .= '|' . $value;
		    }
	    }
	    $hash = null;
	    if (strlen($hash_data) > 0) {
		    $hash = strtoupper(hash("sha512", $hash_data));
	    }
        static::log(array('calculated_hash' => $hash));

        if ($request->response_code!=0) {
            $transaction->setNote('Transaction Failed');
            $status = $transaction::STATUS_FAILED;

        } elseif ($hash!=$request->hash) {
            $transaction->setNote('sha512 hash verification failed');
            $status = $transaction::STATUS_FAILED;

        } elseif ($request->amount!=$transaction->getValue()) {
            $transaction->setNote('request amount and transaction amount not equal');
            $status = $transaction::STATUS_FAILED;
        }
        if ($transaction::STATUS_FAILED === $status) {
            $this->setDetail('error_message', $request->error_Message, 'Error message');
        }

        $transaction->setStatus($status);
    }

    /**
     * Check - payment method is configured or not
     *
     * @param \XLite\Model\Payment\Method $method Payment method
     *
     * @return boolean
     */
    public function isConfigured(\XLite\Model\Payment\Method $method)
    {
        return parent::isConfigured($method)
            && $method->getSetting('api_key')
            && $method->getSetting('salt');
    }

    /**
     * Get return type
     *
     * @return string
     */
    public function getReturnType()
    {
        return self::RETURN_TYPE_HTML_REDIRECT;
    }

    /**
     * Returns the list of settings available for this payment processor
     *
     * @return array
     */
    public function getAvailableSettings()
    {
        return array(
            'api_key',
            'salt',
            'mode',
            'prefix',
        );
    }

    /**
     * Get return request owner transaction or null
     *
     * @return \XLite\Model\Payment\Transaction
     */
    public function getReturnOwnerTransaction()
    {
        return \XLite\Core\Request::getInstance()->udf1
            ? \XLite\Core\Database::getRepo('XLite\Model\Payment\Transaction')->findOneByPublicTxnId(\XLite\Core\Request::getInstance()->udf1)
            : null;
    }

    /**
     * Get payment method admin zone icon URL
     *
     * @param \XLite\Model\Payment\Method $method Payment method
     *
     * @return string
     */
    public function getAdminIconURL(\XLite\Model\Payment\Method $method)
    {
        return true;
    }
	/**
	 * Get return URL
	 *
	 * @return string
	 */
	protected function getBasispayReturnURL()
	{
		return \Includes\Utils\URLManager::getShopURL(
			\XLite\Core\Converter::buildURL(
				'payment_return',
				'',
				array(),
				\XLite::getCustomerScript()
			),
			false
		);
	}
    /**
     * Get redirect form URL
     *
     * @return string
     */
    protected function getFormURL()
    {
        return $this->getSetting('mode')=='test' ? static::TEST_FORM_URL : static::PRODUCTION_FORM_URL;
    }

    /**
     * Return Country field value. if no country defined we should use '' value
     *
     * @param \XLite\Model\Address $address Address model (could be shipping or billing address)
     *
     * @return string
     */
    protected function getCountryField($address)
    {
        return $address->getCountry()
            ? $address->getCountry()->getCode3()
            : '';
    }

    /**
     * Return formatted price.
     *
     * @param float $price Price value
     *
     * @return string
     */
    protected function getFormattedPrice($price)
    {
        return sprintf('%.2f', round((double)($price) + 0.00000000001, 2));
    }

    /**
     * Get redirect form fields list
     *
     * @return array
     */
    protected function getFormFields()
    {
        $fields = array(
            'api_key'         => $this->getSetting('api_key'),
            'order_id'       => $this->getTransactionId(),
            'amount'      => $this->getFormattedPrice($this->transaction->getValue()),
            'description' => 'BasispayV2_xcartV5.3.1: Paying invoice Order: #' . $this->getTransactionId().' using xcart plugin',
            'name'        => $this->getProfile()->getBillingAddress()->getFirstname().' ' .$this->getProfile()->getBillingAddress()->getLastname(),
            'email'       => $this->getProfile()->getLogin(),
            'address_line_1'    => $this->getProfile()->getBillingAddress()->getStreet(),
            'city'        => $this->getProfile()->getBillingAddress()->getCity(),
            'state'       => $this->getProfile()->getBillingAddress()->getState()->getState(),
            'country'     => $this->getCountryField($this->getProfile()->getBillingAddress()),
            'currency'    => 'INR',
            'zip_code'     => $this->getProfile()->getBillingAddress()->getZipcode(),
            'phone'       => $this->getProfile()->getBillingAddress()->getPhone(),
            'return_url'      => $this->getBasispayReturnURL(),
            'udf1'        => $this->transaction->getPublicTxnId(),
        );
	    $fields['mode'] = 'TEST';
	    if($this->getSetting('mode') == 'live'){
		    $fields['mode'] = 'LIVE';
	    }
	    $hash_columns = [
		    'address_line_1',
		    'address_line_2',
		    'amount',
		    'api_key',
		    'city',
		    'country',
		    'currency',
		    'description',
		    'email',
		    'mode',
		    'name',
		    'order_id',
		    'phone',
		    'return_url',
		    'state',
		    'udf1',
		    'udf2',
		    'udf3',
		    'udf4',
		    'udf5',
		    'zip_code',
	    ];

	    ksort($hash_columns);
	    $hash_data = $this->getSetting('salt');
	    foreach ($hash_columns as $column) {
		    if (isset($fields[$column])) {
			    if (strlen($fields[$column]) > 0) {
				    $hash_data .= '|' . trim($fields[$column]);
			    }
		    }
	    }
	    $hash = null;
	    if (strlen($hash_data) > 0) {
		    $hash = strtoupper(hash("sha512", $hash_data));
	    }

        $fields['hash'] = $hash;

        static::log(array('form_fields' => $fields));

        return $fields;
    }

    /**
     * Logging the data under Basispay
     * Available if developer_mode is on in the config file
     *
     * @param mixed $data
     *
     * @return void
     */
    protected static function log($data)
    {
        if (LC_DEVELOPER_MODE) {
            \XLite\Logger::logCustom('Basispay', $data);
        }
    }
}
