<?php


namespace BasisPay\Basispay\Model;

use Magento\Sales\Api\Data\TransactionInterface;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\Framework\Data\Form\FormKey;

class Basispay extends \Magento\Payment\Model\Method\AbstractMethod {

    const PAYMENT_BASISPAY_CODE = 'basispay';
    protected $_code = self::PAYMENT_BASISPAY_CODE;

    /**
     *
     * @var \Magento\Framework\UrlInterface 
     */
    protected $_urlBuilder;
    protected $_supportedCurrencyCodes = array('INR');
    
    private $checkoutSession;

    /**
     * 
     * @param \Magento\Framework\Model\Context $context
     * @param \Magento\Framework\Registry $registry
     * @param \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory
     * @param \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory
     * @param \Magento\Payment\Helper\Data $paymentData
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Payment\Model\Method\Logger $logger
     * @param \Magento\Framework\UrlInterface $urlBuilder
     * @param \Magento\Framework\Model\ResourceModel\AbstractResource $resource
     * @param \Magento\Framework\Data\Collection\AbstractDb $resourceCollection
     * @param array $data
     */
      public function __construct(
        \Magento\Framework\Model\Context $context,
        \Magento\Framework\Registry $registry,
        \Magento\Framework\Api\ExtensionAttributesFactory $extensionFactory,
        \Magento\Framework\Api\AttributeValueFactory $customAttributeFactory,
        \Magento\Payment\Helper\Data $paymentData,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Payment\Model\Method\Logger $logger,
        \BasisPay\Basispay\Helper\Basispay $helper,
        \Magento\Sales\Model\Order\Email\Sender\OrderSender $orderSender,
        \Magento\Framework\HTTP\ZendClientFactory $httpClientFactory,
        \Magento\Checkout\Model\Session $checkoutSession      
              
    ) {
        $this->helper = $helper;
        $this->orderSender = $orderSender;
        $this->httpClientFactory = $httpClientFactory;
        $this->checkoutSession = $checkoutSession;

        parent::__construct(
            $context,
            $registry,
            $extensionFactory,
            $customAttributeFactory,
            $paymentData,
            $scopeConfig,
            $logger
        );

    }

    public function canUseForCurrency($currencyCode) {
        if (!in_array($currencyCode, $this->_supportedCurrencyCodes)) {
            return false;
        }
        return true;
    }

    public function getRedirectUrl() {
        return $this->helper->getUrl($this->getConfigData('redirect_url'));
    }

    public function getReturnUrl() {
        return $this->helper->getUrl($this->getConfigData('return_url'));
    }

    /**
     * Return url according to environment
     * @return string
     */
    public function getCgiUrl() {
        $env = $this->getConfigData('environment');
        if ($env === 'live') {
            return $this->getConfigData('production_url');
        }
        return $this->getConfigData('sandbox_url');
    }

    public function buildCheckoutRequest() {
        $order = $this->checkoutSession->getLastRealOrder();
        $billing_address = $order->getBillingAddress();

        $params = array();
        $params["api_key"] = $this->getConfigData("merchant_key");
        $params["order_id"] = substr(hash('sha256', mt_rand() . microtime()), 0, 20);
        $params["amount"] = round($order->getBaseGrandTotal(), 2);
        $params["description"] = $this->checkoutSession->getLastRealOrderId();
        $params["name"] = $billing_address->getFirstName().' '.$billing_address->getLastname();
        $params["city"]                 = $billing_address->getCity();
        $params["state"]                = $billing_address->getRegion();
        $params["zip_code"]                  = $billing_address->getPostcode();
        if($billing_address->getCountryId() == 'IN'){
            $country = 'IND';
        }else{
            $country = $billing_address->getCountryId();
        }
        $params["country"]              = $country;
        $params["address_line_1"]       = $billing_address->getStreet()[0];
        $params["address_line_2"]       = !empty($billing_address->getStreet()[1]) ? $billing_address->getStreet()[1] : $billing_address->getStreet()[0];
        $params["currency"]             = 'INR';
        $params["email"] = $order->getCustomerEmail();
        $params["phone"] = $billing_address->getTelephone();

        $mode = $this->getConfigData('environment');
        if ($mode === 'live') {
            $mode = 'LIVE';
        }else{
            $mode = 'TEST';
        }
        $params["mode"] = $mode;
        $params["return_url"] = $this->getReturnUrl();

        $params['form_key'] = $_COOKIE['form_key'];
        $params["hash"] = $this->generateHash($params);
        return $params;
    }

    public function generateHash($data) {
        $private_salt = $this->getConfigData('salt');

        $hash_columns = [
            'address_line_1',
            'address_line_2',
            'amount',
            'api_key',
            'city',
            'country',
            'currency',
            'description',
            'email',
            'form_key',
            'mode',
            'name',
            'order_id',
            'phone',
            'return_url',
            'state',
            'udf1',
            'udf2',
            'udf3',
            'udf4',
            'udf5',
            'zip_code',
        ];
        ksort($hash_columns);
        $hash_data = $private_salt;

        foreach ($hash_columns as $column) {
            if (isset($data[$column])) {
                if (strlen($data[$column]) > 0) {
                    $hash_data .= '|' . trim($data[$column]);
                }
            }
        }

        if (strlen($hash_data) > 0) {
            $hash = strtoupper(hash("sha512", $hash_data));
        }
        return $hash;
    }

    //validate response
    public function validateResponse($returnParams) {
	    if ($returnParams['response_code'] != 0 ) {
            return false;
        }
        $hash_columns = $returnParams;
        unset($hash_columns['hash']);
        unset($hash_columns['form_key']);

        ksort($hash_columns);
        $private_salt = $this->getConfigData('salt');
        $hash_data = $private_salt;

        foreach ($hash_columns as $key=>$value) {
            if (strlen($value) > 0) {
                $hash_data .= '|' . $value;
            }
        }

	    $hash = null;
        if (strlen($hash_data) > 0) {
            $hash = strtoupper(hash("sha512", $hash_data));
        }

        if ($returnParams['hash'] != $hash) {
	        return false;
        }
	    return true;
    }

    public function postProcessing(\Magento\Sales\Model\Order $order,
            \Magento\Framework\DataObject $payment, $response) {
        
        $payment->setTransactionId($response['transaction_id']);
        $payment->setTransactionAdditionalInfo('basispay_payment_method',$response['payment_mode']);
        $payment->setAdditionalInformation('basispay_response_message', $response['response_message']);
        $payment->addTransaction(TransactionInterface::TYPE_ORDER);
        $payment->setIsTransactionClosed(0);
        $payment->place();
        $order->setStatus('processing');
        $order->save();
    }

}
